<?php
/*
Plugin Name: WP-EVP (WordPress EasyVideoPlayer Plugin)
Plugin URI: http://webactix.com/wp-evp-plugin/
Description: Plugin to easily embed EasyVideoPlayer videos into WordPress posts!
Date: 2011, January, 27
Author: WebActix
Author URI: http://webactix.com
Version: 1.4
*/

/**
 * extract_tags()
 * Extract specific HTML tags and their attributes from a string.
 *
 * You can either specify one tag, an array of tag names, or a regular expression that matches the tag name(s). 
 * If multiple tags are specified you must also set the $selfclosing parameter and it must be the same for 
 * all specified tags (so you can't extract both normal and self-closing tags in one go).
 * 
 * The function returns a numerically indexed array of extracted tags. Each entry is an associative array
 * with these keys :
 * 	tag_name	- the name of the extracted tag, e.g. "a" or "img".
 *	offset		- the numberic offset of the first character of the tag within the HTML source.
 *	contents	- the inner HTML of the tag. This is always empty for self-closing tags.
 *	attributes	- a name -> value array of the tag's attributes, or an empty array if the tag has none.
 *	full_tag	- the entire matched tag, e.g. '<a href="http://example.com">example.com</a>'. This key 
 *		          will only be present if you set $return_the_entire_tag to true.	   
 *
 * @param string $html The HTML code to search for tags.
 * @param string|array $tag The tag(s) to extract.							 
 * @param bool $selfclosing	Whether the tag is self-closing or not. Setting it to null will force the script to try and make an educated guess. 
 * @param bool $return_the_entire_tag Return the entire matched tag in 'full_tag' key of the results array.  
 * @param string $charset The character set of the HTML code. Defaults to ISO-8859-1.
 *
 * @return array An array of extracted tags, or an empty array if no matching tags were found. 
 */
function evp_extract_tags( $html, $tag, $selfclosing = null, $return_the_entire_tag = false, $charset = 'ISO-8859-1' ){
 
	if ( is_array($tag) ){
		$tag = implode('|', $tag);
	}
 
	//If the user didn't specify if $tag is a self-closing tag we try to auto-detect it
	//by checking against a list of known self-closing tags.
	$selfclosing_tags = array( 'area', 'base', 'basefont', 'br', 'hr', 'input', 'img', 'link', 'meta', 'col', 'param' );
	if ( is_null($selfclosing) ){
		$selfclosing = in_array( $tag, $selfclosing_tags );
	}
 
	//The regexp is different for normal and self-closing tags because I can't figure out 
	//how to make a sufficiently robust unified one.
	if ( $selfclosing ){
		$tag_pattern = 
			'@<(?P<tag>'.$tag.')			# <tag
			(?P<attributes>\s[^>]+)?		# attributes, if any
			\s*/?>					# /> or just >, being lenient here 
			@xsi';
	} else {
		$tag_pattern = 
			'@<(?P<tag>'.$tag.')			# <tag
			(?P<attributes>\s[^>]+)?		# attributes, if any
			\s*>					# >
			(?P<contents>.*?)			# tag contents
			</(?P=tag)>				# the closing </tag>
			@xsi';
	}
 
	$attribute_pattern = 
		'@
		(?P<name>\w+)							# attribute name
		\s*=\s*
		(
			(?P<quote>[\"\'])(?P<value_quoted>.*?)(?P=quote)	# a quoted value
			|							# or
			(?P<value_unquoted>[^\s"\']+?)(?:\s+|$)			# an unquoted value (terminated by whitespace or EOF) 
		)
		@xsi';
 
	//Find all tags 
	if ( !preg_match_all($tag_pattern, $html, $matches, PREG_SET_ORDER | PREG_OFFSET_CAPTURE ) ){
		//Return an empty array if we didn't find anything
		return array();
	}
 
	$tags = array();
	foreach ($matches as $match){
 
		//Parse tag attributes, if any
		$attributes = array();
		if ( !empty($match['attributes'][0]) ){ 
 
			if ( preg_match_all( $attribute_pattern, $match['attributes'][0], $attribute_data, PREG_SET_ORDER ) ){
				//Turn the attribute data into a name->value array
				foreach($attribute_data as $attr){
					if( !empty($attr['value_quoted']) ){
						$value = $attr['value_quoted'];
					} else if( !empty($attr['value_unquoted']) ){
						$value = $attr['value_unquoted'];
					} else {
						$value = '';
					}
 
					//Passing the value through html_entity_decode is handy when you want
					//to extract link URLs or something like that. You might want to remove
					//or modify this call if it doesn't fit your situation.
					$value = html_entity_decode( $value, ENT_QUOTES, $charset );
 
					$attributes[$attr['name']] = $value;
				}
			}
 
		}
 
		$tag = array(
			'tag_name' => $match['tag'][0],
			'offset' => $match[0][1], 
			'contents' => !empty($match['contents'])?$match['contents'][0]:'', //empty for self-closing tags
			'attributes' => $attributes, 
		);
		if ( $return_the_entire_tag ){
			$tag['full_tag'] = $match[0][0]; 			
		}
 
		$tags[] = $tag;
	}
 
	return $tags;
}

// Load the custom TinyMCE buttons
function evp_tinymce($buttons) {
	array_push($buttons, 'wpevp');
	return $buttons;
}
add_filter('mce_buttons', 'evp_tinymce');

// Load the custom TinyMCE plugin
function evp_tinymce_plugin($plugins) {
	$plugins['wpevp'] = plugins_url('/wp-evp/scripts/editor_plugin.js');
	return $plugins;
}
add_filter('mce_external_plugins', 'evp_tinymce_plugin');

// Break the browser cache of TinyMCE
function evp_tinymce_version($version) {
	return $version . '-wp-evp';
}
add_filter('tiny_mce_version', 'evp_tinymce_version');

// Are we an admin and do we need to load things up?
if(is_admin()) {
	if(in_array(basename($_SERVER['PHP_SELF']), array('post-new.php', 'page-new.php', 'post.php', 'page.php'))) {
		wp_enqueue_script('phpjs', plugins_url('/wp-evp/scripts/phpjs.js'));
	}
}

// And for non-tinyMCE and general functions
function evp_jsfunctions() {
	?>
	<script type="text/javascript">
	function WPEVPPopup() {
	  var evpcode = prompt('Please enter your EasyVideoPlayer code here. Audio files and lightboxes will work also!');
	  if(typeof(evpcode) !== 'undefined' && evpcode.length > 0) {
		var url = '<?php echo plugins_url('/wp-evp/images/placeholder.png'); ?>';
		
		evpcode = evpcode.replace('<!--', '').replace('//-->', '');
		
		evpcode = base64_encode(evpcode);
		
		var html = '<!-- WP EVP Video - DO NOT MODIFY --><img class="wpevp-container" src="'+url+'" style="display: block; width: 400px; height: 200px;" alt="'+evpcode+'" /><!-- END WP EVP Video - DO NOT MODIFY -->';
		edInsertContent(edCanvas, html);
	  }
	}
	jQuery(document).ready(function() {
		/* Add to the blank editor */
		jQuery('#ed_toolbar').append('<input type="button" class="ed_button" onclick="WPEVPPopup()" title="Click" value="Embed an EVP video" />');
	});
	</script>
	<?php
}
add_action('edit_form_advanced', 'evp_jsfunctions');
add_action('edit_page_form', 'evp_jsfunctions');

/*
 * Actually convert that into real code
 */
function evp_filter($content) {	
	$images = evp_extract_tags($content, 'img', null, true);

	if(!empty($images)) {
		foreach($images as $img) {
			$alt = $img['attributes']['alt'];
			$evpcode = '';
			
			if(!empty($alt)) {
				$evpcode = @base64_decode($alt);
			}

			if(stripos($evpcode, '<div id="evp') === false) { // If it doesn't contain valid EVP code, it's a normal image!
				// Do nothing
			} else {
			//$evpcode = str_replace(array('<!--', '//-->'), '', $evpcode);
			  $content = str_replace($img['full_tag'], $evpcode, $content);
			  //$content = str_replace($img['full_tag'], $evpcode, $content);
			}
		}
	}		
	$content = str_replace(array('<!-- WP EVP Video - DO NOT MODIFY -->', '<!-- END WP EVP Video - DO NOT MODIFY -->'), '', $content);

	return $content;
}

add_filter('the_content', 'evp_filter', 9999);
add_filter('the_excerpt', 'evp_filter', 9999);
?>
